function [alt] = isa_search(rho_target)

%   ---------------------------------------------------------------------------------------------------------------------
%   FUNCTION DESCRIPTION: Reverse lookup of altitude from density in the Standard Atmosphere Table
%                         Matlab's ISA Table is only good up to 20,000 meters.
%
%   ---- EXAMPLE ----
%   [alt] = isa_search(1.0);
%
%   ---- USER INPUTS ----
%   rho_target                    % Fully inflated balloon volume (Kg/m^3)
%
%   ---- OUTPUTS ----
%   alt                           % Altitude that corresponds to an atmospheric density of rho_target (meters)
%
%   ---------------------------------------------------------------------------------------------------------------------

alt = 0;    % initialize return variable alt

% Check if Aerospace Toolbox Installed - which includes the atmosisa() function.
if (license('test','aerospace_toolbox') == 0)
    disp('You do not have the Aerospace Toolbox Installed');
    disp('You can still use the Standard Atmosphere Table by Hand');
    disp(['Look up the row that corresponds to a density of: ', num2str(rho_target), ' Kg/m^3']);
    disp('to find the corresponding altitude, ambient pressure and temperature.');
    return;
end

% Special Case - Alt <= 0 meters - Assume we are SSL or above
if (rho_target > 1.225)
    rho_target = 1.225;
    alt = 0;
    return;
end

% Special Case - Alt >= 20,000 meters
if (rho_target <= 0.0880) % Special case
    disp('ERROR: Atmospheric Density < 0.0880 Kg/m^3');
    disp('Matlab atmosisa() function does not support lookup above 20,000 meters');
    disp('Matlab R2023a allows for extended table up to 84,852 meters using atmosisa(rho_target,extended=true)');
    disp('You can still use the Standard Atmosphere Table by Hand');
    disp(['Look up the row that corresponds to a density of: ', num2str(rho_target), ' Kg/m^3']);
    disp('to find the corresponding altitude, ambient pressure and temperature.');
    alt = 20e3;
    return;
end

% Initialize Search Variables
alt = 10e3;
delta_alt = 10e3;
[~,~,~,rho] = atmosisa(alt);

while ((abs(rho-rho_target) > 0.0001)) 

    % Update rho based on alt guess
    [~,~,~,rho] = atmosisa(alt);
    
    % if not close enough guess new altitude
    delta_alt = (delta_alt/2);
    if (rho <= rho_target) % go down in altitude
        alt = alt - delta_alt;
    elseif (rho > rho_target) % go up in altitude
        alt = alt + delta_alt;
    end

end

end