%   ---------------------------------------------------------------------------------------------------------------------
%   SCRIPT DESCRIPTION: Example script to show how to use b_calc and isa_search to compute
%                       key balloon performance metrics.
%
%   ---- b_calc() FUNCTION CALL SYNTAX ----
%   [h_i, h_f, p_f, delta_p_f] = b_calc(R_gas, D_launch_site, P_launch_site, T_launch_site, m_f, m_b, m_p, v_balloon)
%
%   ---- USER INPUTS ----
%   R_gas                         % Gas Constant in J/(Kg*K) (2077 for Helium, 4124 for Hydrogen)
%   D_launch_site                 % Atmospheric Density at launch site in Kg/m^3 (1.225 for SSL)
%   P_launch_site                 % Atmospheric Pressure at launch site in Pa (101325 for SSL)
%   T_launch_site                 % Atmospheric Temperature at launch site in Kelvin (288 for SSL)
%   m_f                           % Desired Free lift weight (Kg) (0.0024 Kg for example)
%   m_b                           % Mass of balloon material (Kg) (0.037 Kg for example)
%   m_p                           % Mass of Payload (Kg) (0.015 Kg for example)
%   v_balloon                     % Fully inflated balloon volume (m^3) (0.1m^3 or 0.25m^3 for example)
%
%   ---- OUTPUTS ----
%   h_i                           % Fully inflated balloon altitude (meters)
%   h_f                           % Final balloon float altitude (meters)
%   p_f                           % Balloon's internal Pressure at floating altitude (Pa)
%   delta_p_f                     % Pressure difference between balloon's internal pressure and ambient pressure (Pa)
%
%   ---------------------------------------------------------------------------------------------------------------------

clear
clc

% ---- Plot Range & Fidelity ----
steps = 101;
min_mass = 0.005;
max_mass = 0.1;
mass = zeros(1,steps);

% ---- initialize return values ----
h_i = zeros(1,steps);
h_f = zeros(1,steps);
p_f = zeros(1,steps);
delta_p_f = zeros(1,steps);

% ---- Compute Balloon Volume ----
for n = 1:steps
   mass(1,n) = n*((max_mass - min_mass)/(steps-1));

   % --- Enter your modified values into this function call ---
   [h_i(1,n), h_f(1,n), p_f(1,n), delta_p_f(1,n)] = b_calc(2077, 1.225, 101325, 288, 0.0024, 0.037, mass(1,n), 0.1);
end

% ---- Plot Data ----
subplot(2,2,1)
plot(mass, h_i)
title('Fully Inflated Altitude')
xlabel('Payload Mass (Kg)')
ylabel('Altitude (m)')
grid on;

subplot(2,2,2)
plot(mass, h_f)
title('Final Float Altitude')
xlabel('Payload Mass (Kg)')
ylabel('Altitude (m)')
grid on;

subplot(2,2,3)
plot(mass, p_f)
title('Internal Pressure at Floating Altitude')
xlabel('Payload Mass (Kg)')
ylabel('Balloon Internal Pressure (Pa)')
grid on;

subplot(2,2,4)
plot(mass, delta_p_f)
title('Pressure Difference at Float Altitude')
xlabel('Payload Mass (Kg)')
ylabel('Balloon Internal Pressure - Ambient Pressure (Pa)')
grid on;