function [h_i, h_f, p_f, delta_p_f] = b_calc(R_gas, D_launch_site, P_launch_site, T_launch_site, m_f, m_b, m_p, v_balloon)

%   ---------------------------------------------------------------------------------------------------------------------
%   FUNCTION DESCRIPTION: Compute key balloon performance metrics
%
%   ---- EXAMPLE ----
%   [h_i, h_f, p_f, delta_p_f] = b_calc(2077, 1.225, 101325, 288, 0.0024, 0.037, 0.015, 0.1);
%
%   ---- USER INPUTS ----
%   R_gas                         % Gas Constant in J/(Kg*K) (2077 for Helium, 4124 for Hydrogen)
%   D_launch_site                 % Atmospheric Density at launch site in Kg/m^3 (1.225 for SSL)
%   P_launch_site                 % Atmospheric Pressure at launch site in Pa (101325 for SSL)
%   T_launch_site                 % Atmospheric Temperature at launch site in Kelvin (288 for SSL)
%   m_f                           % Desired Free lift weight (Kg) (0.0024 Kg in example)
%   m_b                           % Mass of balloon material (Kg) (0.037 Kg in example)
%   m_p                           % Mass of Payload (Kg) (0.015 Kg in example)
%   v_balloon                     % Fully inflated balloon volume (m^3) (0.1 m^3 in example)
%
%   ---- OUTPUTS ----
%   h_i                           % Fully inflated balloon altitude (meters)
%   h_f                           % Final balloon float altitude (meters)
%   p_f                           % Balloon's internal Pressure at floating altitude (Pa)
%   delta_p_f                     % Pressure difference between balloon's internal pressure and ambient pressure (Pa)
%
%   ---------------------------------------------------------------------------------------------------------------------

% ---- STEP 1: Compute Helium required to meet Free Lift Weight Design Criteria ----
m_gas = (m_f+m_b+m_p)/((R_gas*T_launch_site*D_launch_site/P_launch_site)-1);

% ---- STEP 2: Compute Fully Inflated Altitude ----
D_lookup = (m_f+m_b+m_p+m_gas)/v_balloon;
h_i = isa_search(D_lookup);

% ---- STEP 3: Compute Float Altitude ----
D_lookup = (m_b+m_p+m_gas)/v_balloon;
h_f = isa_search(D_lookup);
[T_ambient, ~, P_ambient, ~] = atmosisa(h_f);
p_f = (m_gas*R_gas*T_ambient)/v_balloon;
delta_p_f = p_f - P_ambient;

end